const flowCache = new Map();

jQuery(document).ready(function ($) {
  // Verificar se ajax_object está disponível (silencioso)
  if (typeof ajax_object === 'undefined') {
    console.error('❌ ajax_object NÃO está disponível!');
  }
  
  // Observar quando elementos são adicionados ao modal para detectar o botão
  const observer = new MutationObserver(function(mutations) {
    mutations.forEach(function(mutation) {
      if (mutation.addedNodes.length > 0) {
        mutation.addedNodes.forEach(function(node) {
          if (node.nodeType === 1) { // Element node
            const $node = $(node);
            // Verificar se o botão foi adicionado
            // Botão detectado via MutationObserver (sem log para não poluir o console)
            if ($node.is('#upgram-clear-posts-cache') || $node.find('#upgram-clear-posts-cache').length > 0) {
              // Botão encontrado e pronto para uso
            }
          }
        });
      }
    });
  });
  
  // Observar mudanças no modal-container
  const $modalContainer = $('#modal-container');
  if ($modalContainer.length > 0) {
    observer.observe($modalContainer[0], {
      childList: true,
      subtree: true
    });
        // MutationObserver configurado para #modal-container
  } else {
    // Se o modal ainda não existe, tentar novamente depois
    setTimeout(function() {
      const $modalContainerRetry = $('#modal-container');
      if ($modalContainerRetry.length > 0) {
        observer.observe($modalContainerRetry[0], {
          childList: true,
          subtree: true
        });
        // MutationObserver configurado para #modal-container (retry)
      }
    }, 1000);
  }
  $("body").append(
    '<div class="modal fade upgram-root" id="modal-container" tabindex="-1" aria-labelledby="modal-container-label" role="dialog" aria-modal="true"></div>'
  );
  
  // Corrigir aria-hidden quando o modal for aberto/fechado
  $(document).on('show.bs.modal', '#modal-container', function() {
    const $modal = $(this);
    // Remover aria-hidden antes de mostrar
    $modal.removeAttr('aria-hidden');
  });
  
  $(document).on('shown.bs.modal', '#modal-container', function() {
    const $modal = $(this);
    // Garantir que não tenha aria-hidden quando visível
    $modal.removeAttr('aria-hidden');
    $modal.attr('aria-modal', 'true');
    // Remover tabindex para evitar problemas de foco
    $modal.removeAttr('tabindex');
  });
  
  $(document).on('hide.bs.modal', '#modal-container', function() {
    const $modal = $(this);
    // Definir aria-hidden antes de esconder
    $modal.attr('aria-hidden', 'true');
    $modal.removeAttr('aria-modal');
  });
  
  $(document).on('hidden.bs.modal', '#modal-container', function() {
    const $modal = $(this);
    // Garantir que tenha aria-hidden quando oculto
    $modal.attr('aria-hidden', 'true');
    $modal.removeAttr('aria-modal');
  });
  
  $("body").append(
    `
    <div id="upgram-toast-container" class="position-fixed top-0 end-0 p-3" style="z-index: 999999; display: none;">
      <div id="errorToast" class="toast align-items-center text-white bg-danger border-0" role="alert" aria-live="assertive" aria-atomic="true">
          <div class="d-flex align-items-center justify-content-between">
              <div class="toast-body"></div>
              <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
          </div>
      </div>
      <div id="successToast" class="toast align-items-center text-white bg-success border-0" role="alert" aria-live="assertive" aria-atomic="true">
          <div class="d-flex align-items-center justify-content-between">
              <div class="toast-body"></div>
              <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
          </div>
      </div>
    </div>
    `
  );

  // No mobile, adiar verificação de licença para não bloquear carregamento
  // Só verificar quando realmente necessário (quando usuário interagir)
  const isMobile = /Mobile|Android|iPhone|iPad/.test(navigator.userAgent);
  
  // Cache para evitar múltiplas verificações simultâneas
  // No mobile, cache mais longo e timeout menor para priorizar carregamento
  let licenseCheckPromise = null;
  let lastLicenseCheck = null;
  const LICENSE_CHECK_CACHE_TIME = isMobile ? 120000 : 60000; // Mobile: 2 minutos, Desktop: 1 minuto
  const LICENSE_CHECK_TIMEOUT = isMobile ? 15000 : 20000; // Mobile: 15 segundos, Desktop: 20 segundos

  async function checkLicenseKey() {
    // Se já há uma verificação em andamento, retornar a mesma promise
    if (licenseCheckPromise) {
      return licenseCheckPromise;
    }

    // Se verificamos recentemente (dentro do cache), retornar resultado em cache
    if (lastLicenseCheck && (Date.now() - lastLicenseCheck.timestamp) < LICENSE_CHECK_CACHE_TIME) {
      return Promise.resolve(lastLicenseCheck.result);
    }

    // Criar nova verificação
    licenseCheckPromise = new Promise((resolve, reject) => {
      const timeoutId = setTimeout(() => {
        licenseCheckPromise = null;
        // Em caso de timeout, assumir licença válida temporariamente para não bloquear
        // No mobile, isso é ainda mais importante para não travar a página
        const fallbackResult = {
          success: true,
          data: {
            valid: true,
            message: 'Verificação temporariamente indisponível'
          }
        };
        lastLicenseCheck = {
          result: fallbackResult,
          timestamp: Date.now()
        };
        resolve(fallbackResult); // No mobile, resolver ao invés de rejeitar para não bloquear
      }, LICENSE_CHECK_TIMEOUT);

      $.ajax({
        url: ajax_object.ajax_url,
        type: "POST",
        dataType: "json",
        timeout: LICENSE_CHECK_TIMEOUT, // Timeout otimizado para mobile
        data: {
          action: "check_license_key",
          security: ajax_object.security,
        },
        success: (data) => {
          clearTimeout(timeoutId);
          licenseCheckPromise = null;
          lastLicenseCheck = {
            result: data,
            timestamp: Date.now()
          };
          resolve(data);
        },
        error: (xhr, textStatus, errorThrown) => {
          clearTimeout(timeoutId);
          licenseCheckPromise = null;
          
          // Em caso de erro, assumir licença válida temporariamente para não bloquear
          if (textStatus === 'timeout' || xhr.status === 0) {
            const fallbackResult = {
              success: true,
              data: {
                valid: true,
                message: 'Verificação temporariamente indisponível'
              }
            };
            lastLicenseCheck = {
              result: fallbackResult,
              timestamp: Date.now()
            };
            resolve(fallbackResult);
          } else {
            reject(errorThrown);
          }
        },
      });
    });

    return licenseCheckPromise;
  }
  
  let licenseKeyPromise = null;
  
  // Função para obter promise de licença (lazy loading no mobile)
  function getLicenseKeyPromise() {
    if (!licenseKeyPromise) {
      licenseKeyPromise = checkLicenseKey();
    }
    return licenseKeyPromise;
  }
  
  // No desktop, verificar imediatamente
  // No mobile, só verificar quando necessário (lazy)
  if (!isMobile) {
    licenseKeyPromise = checkLicenseKey();
  }

  // Circuit breaker para evitar múltiplas requisições simultâneas
  // No mobile, ser mais restritivo para priorizar carregamento
  let ajaxRequestCount = 0;
  const MAX_CONCURRENT_REQUESTS = isMobile ? 2 : 3; // Mobile: máximo 2, Desktop: 3
  // Timeout padrão para requisições normais
  const REQUEST_TIMEOUT = isMobile ? 15000 : 20000; // Mobile: 15 segundos, Desktop: 20 segundos
  // Timeout aumentado para requisições de stories (podem demorar mais)
  const STORIES_REQUEST_TIMEOUT = isMobile ? 40000 : 60000; // Mobile: 40 segundos, Desktop: 60 segundos

  function ajax(data) {
    return new Promise((resolve, reject) => {
      // Verificar se há muitas requisições pendentes
      if (ajaxRequestCount >= MAX_CONCURRENT_REQUESTS) {
        if (!isMobile) {
          console.warn('Upgram: Muitas requisições simultâneas, aguardando...');
        }
        const retryDelay = isMobile ? 1000 : 500; // Mobile: aguardar mais tempo
        setTimeout(() => {
          ajax(data).then(resolve).catch(reject);
        }, retryDelay);
        return;
      }

      ajaxRequestCount++;
      
      // Detectar se é requisição de stories para usar timeout maior
      // Verificar tanto em get_more_content quanto em save_selected_posts
      const isStoriesRequest = (data.action === 'load_more_posts' || data.action === 'save_selected_posts') && 
                               (data.service_type === 'visualizacoes_story' || 
                                (typeof data.product_id !== 'undefined' && 
                                 (function() {
                                   // Verificar se o product_id corresponde a stories
                                   // Isso será verificado no backend, mas podemos assumir que é stories
                                   // se a ação for save_selected_posts (que busca stories)
                                   return data.action === 'save_selected_posts';
                                 })()));
      const timeout = isStoriesRequest ? STORIES_REQUEST_TIMEOUT : REQUEST_TIMEOUT;
      
      // Timeout para evitar que a requisição trave indefinidamente
      const timeoutId = setTimeout(() => {
        ajaxRequestCount--;
        reject(new Error('Timeout: A requisição demorou muito para responder. Por favor, tente novamente.'));
      }, timeout);

      $.ajax({
        url: ajax_object.ajax_url,
        type: "POST",
        dataType: "json",
        timeout: timeout, // Timeout do jQuery (aumentado para stories)
        data: {
          ...data,
          security: ajax_object.security,
        },
        success: function (json) {
          clearTimeout(timeoutId);
          ajaxRequestCount--;
          if (json.success) {
            resolve(json.data);
          } else {
            reject(json.data);
            if (json.data && json.data.message) {
              displayErrorToast(json.data.message);
            }
          }
        },
        error: function (xhr, textStatus, errorThrown) {
          clearTimeout(timeoutId);
          ajaxRequestCount--;
          console.error("AJAX ERROR: ", textStatus, errorThrown);
          
          // Tratamento de erros específicos
          let errorMessage = 'Erro ao processar requisição.';
          if (textStatus === 'timeout') {
            errorMessage = 'Timeout: A requisição demorou muito. Por favor, tente novamente.';
          } else if (textStatus === 'abort') {
            errorMessage = 'Requisição cancelada.';
          } else if (xhr.status === 0) {
            errorMessage = 'Erro de conexão. Verifique sua internet.';
          } else if (xhr.status >= 500) {
            errorMessage = 'Erro no servidor. Por favor, tente novamente mais tarde.';
          } else if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
            errorMessage = xhr.responseJSON.data.message;
          }
          
          reject(new Error(errorMessage));
          displayErrorToast(errorMessage);
        },
      });
    });
  }


  // Handler do botão Continuar - Prioridade alta para interceptar antes de outros handlers
  $(document).on("click", "a", function (e) {
    const anchor = e.currentTarget;
    
    // Verificar se é um link com variation_id
    let variationId = null;
    try {
      const source = anchor.href ? new URL(anchor.href).searchParams : null;
      if (source) {
        variationId = source.get("variation_id");
      }
    } catch (err) {
      // Se não conseguir parsear URL, não é nosso link
      return;
    }
    
    if (!variationId) return;

    // Verificar modo de checkout ANTES de qualquer processamento
    const checkoutConfig = window.UpgramCheckoutConfig || {};
    const checkoutMode = checkoutConfig.mode || 'popup';
    const checkoutPageUrl = checkoutConfig.checkoutPageUrl || '';

    // Se modo for 'page', redirecionar imediatamente
    if (checkoutMode === 'page' && checkoutPageUrl) {
      e.preventDefault();
      e.stopPropagation();
      if (typeof e.stopImmediatePropagation === 'function') {
        e.stopImmediatePropagation();
      }

      // Montar URL de redirecionamento com parâmetros necessários
      const url = new URL(checkoutPageUrl);
      url.searchParams.set('variation_id', variationId);
      url.searchParams.set('step', '1');

      // Log temporário para debug
      console.log('[Upgram] mode:', checkoutMode);
      console.log('[Upgram] checkoutPageUrl:', checkoutPageUrl);
      console.log('[Upgram] variationId:', variationId);
      console.log('[Upgram] redirect url:', url.toString());

      // Redirecionar
      window.location.href = url.toString();
      return false;
    }

    // Se modo for 'popup', manter comportamento atual (abrir modal)
    e.preventDefault();
    
    requestAnimationFrame(async () => {
      const state = await getLicenseKeyPromise();
      const isValid = state.data?.valid;
      if (!isValid) {
        alert("Licença inválida ou expirada!");
        return;
      }

      const { html } = await ajax({
        action: "load_initial_modal",
        variation_id: variationId,
      });
      
      // Remover aria-hidden antes de mostrar o modal
      const $modal = $("#modal-container");
      $modal.removeAttr('aria-hidden');
      $modal.attr('aria-modal', 'true');
      
      $modal.html(html).modal("show");
      
      // Garantir que o aria-hidden seja removido após o modal ser mostrado
      setTimeout(function() {
        const $modal = $("#modal-container");
        if ($modal.length) {
          $modal.removeAttr('aria-hidden');
          $modal.attr('aria-modal', 'true');
        }
      }, 100);
      
      // Verificar licença após carregar a modal e bloquear botão se necessário
      checkLicenseKey().then((licenseState) => {
        const isValid = licenseState.data?.valid;
        if (!isValid) {
          const submitButton = $("#modal-container").find('button[type="submit"]');
          submitButton.prop('disabled', true)
            .css('opacity', '0.6')
            .css('cursor', 'not-allowed')
            .attr('title', licenseState.data?.message || 'Licença inválida');
          
          // Adicionar mensagem de erro acima do botão
          const errorMessage = licenseState.data?.message || 'Licença inválida, expirada ou não validada. Por favor, entre em contato com o suporte.';
          if ($("#modal-container").find('.license-error-message').length === 0) {
            submitButton.before(
              '<div class="license-error-message" style="background: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 12px; border-radius: 4px; margin-bottom: 15px; font-size: 14px;">' +
              '<strong>⚠️ Erro:</strong> ' + errorMessage +
              '</div>'
            );
          }
        }
      }).catch((error) => {
        console.error('Erro ao verificar licença:', error);
      });
    });
  });

  // Adicionar loading IMEDIATAMENTE quando o botão Continuar da modal é pressionado
  // Isso garante feedback visual instantâneo antes do submit
  // Inclui tanto botões dentro do form quanto botões com form="instagram-form"
  $(document).on("mousedown touchstart", "#instagram-form button[type='submit'], button[type='submit'][form='instagram-form']", function(e) {
    // Mostrar loading instantaneamente (antes do submit ser processado)
    // Usar requestAnimationFrame para evitar reflow forçado
    requestAnimationFrame(() => {
      // O seletor [id^='continue-button-spinner'] pega todos os spinners (incluindo spinner2)
      $("[id^='continue-button-icon']").hide();
      $("[id^='continue-button-spinner']").show();
      $("[id^='continue-button-spinner2']").show();
    });
  });

  $(document).on("submit", "#instagram-form", async (e) => {
    e.preventDefault();
    
    // Garantir que loading está visível (caso mousedown não tenha disparado)
    $("[id^='continue-button-icon']").hide();
    $("[id^='continue-button-spinner']").show();
    
    // Verificar licença antes de permitir o submit
    try {
      const licenseState = await getLicenseKeyPromise();
      const isValid = licenseState.data?.valid;
      if (!isValid) {
        const errorMessage = licenseState.data?.message || 'Licença inválida, expirada ou não validada. Por favor, entre em contato com o suporte.';
        displayErrorToast(errorMessage);
        // Restaurar ícone se houver erro
        $("[id^='continue-button-icon']").show();
        $("[id^='continue-button-spinner']").hide();
        return;
      }
    } catch (error) {
      displayErrorToast('Erro ao verificar licença. Por favor, entre em contato com o suporte.');
      // Restaurar ícone se houver erro
      $("[id^='continue-button-icon']").show();
      $("[id^='continue-button-spinner']").hide();
      return;
    }
    
    const formData = new FormData(e.target);
    flowCache.set("startFormData", formData);
    console.log(
      "submitted",
      Array.from(formData),
      Object.fromEntries(formData)
    );
    if (formData.get("skip") === "false") {
      $("#modal-container").find("#instagram-form").hide();
      $("#modal-container").find("#instructions").show();
      $("#modal-container").find('input[name="skip"]').val("true");
      // Restaurar ícone quando mostra instruções
      $("[id^='continue-button-icon']").show();
      $("[id^='continue-button-spinner']").hide();
      return;
    }

    const data = Object.fromEntries(formData.entries());

    // Loading já deve estar visível do mousedown, mas garantir
    $("[id^='continue-button-icon']").hide();
    $("[id^='continue-button-spinner']").show();

    try {
      const { html } = await ajax({
        action: "submit_to_cart",
        ...data,
      });

      $("#modal-container").find("#instructions")?.hide();
      $("#modal-container").find("#initialModal").hide();
      $("#modal-container").find("#instagram-form").show(); // undo hide if not skip before
      
      $("#modal-container").append(html);

      // Carregar métricas se houver posts no HTML inserido
      // Verificar múltiplas vezes para garantir que o DOM foi renderizado

      // HANDLE CPF VISIBILITY
      if (
        $('input[name="payment_method"][value="paghiper_pix"]').is(":checked")
      ) {
        $("#upgram-cpf").removeClass("d-none");
      }
      $('input[name="payment_method"]').change(function () {
        if ($(this).val() === "paghiper_pix" && $(this).is(":checked")) {
          $("#upgram-cpf").removeClass("d-none");
        } else {
          $("#upgram-cpf").addClass("d-none");
        }
      });
    } finally {
      $("[id^='continue-button-icon']").show();
      $("[id^='continue-button-spinner']").hide();
    }
  });

  let loading = false;
  // bind load more listener in case it's curtidas
  // Botão Limpar Cache - Usar delegação de eventos para funcionar com elementos dinâmicos
  $(document).on("click", "#upgram-clear-posts-cache", function (e) {
    e.preventDefault();
    e.stopPropagation();
    
    const $btn = $(this);
    const $btnText = $btn.find('.upgram-clear-cache-btn-text');
    const $btnLoading = $btn.find('.upgram-clear-cache-btn-loading');
    const username = $btn.data('username');
    const productId = $btn.data('product-id');
    const serviceType = $btn.data('service-type');
    
    if (!username || !productId) {
      console.error('Dados inválidos:', { username, productId });
      displayErrorToast('Dados inválidos para limpar cache');
      return;
    }
    
    if (typeof ajax_object === 'undefined' || !ajax_object.ajax_url || !ajax_object.security) {
      console.error('ajax_object não está definido ou está incompleto');
      displayErrorToast('Erro de configuração. Recarregue a página e tente novamente.');
      return;
    }
    
    // Desabilitar botão e mostrar loading
    $btn.prop('disabled', true);
    $btnText.hide();
    $btnLoading.show();
    
    // Mostrar loading na área de posts
    const $postsContainer = $('#upgram-posts');
    const $loadingState = $('.loading-state');
    
    // Garantir que o container de loading está visível
    if ($loadingState.length) {
      $loadingState.removeClass('d-none').show();
    }
    
    // Esconder posts com animação
    $postsContainer.fadeOut(200, function() {
      // Após esconder, garantir que loading está visível
      if ($loadingState.length) {
        $loadingState.removeClass('d-none').show();
      }
    });
    
    $.ajax({
      url: ajax_object.ajax_url,
      type: "POST",
      dataType: "json",
      timeout: 60000, // 60 segundos de timeout
      data: {
        action: "clear_posts_cache",
        security: ajax_object.security,
        username: username,
        product_id: productId,
        service_type: serviceType
      },
      beforeSend: function() {
        // Requisição AJAX iniciada
      },
      success: function (response) {
        console.log('Resposta clear_posts_cache recebida:', response);
        if (response && response.success && response.data && response.data.html) {
          // Substituir conteúdo dos posts completamente
          $postsContainer.html(response.data.html);
          
          // Re-inicializar Alpine.js se necessário (para stories)
          if (typeof Alpine !== 'undefined') {
            Alpine.initTree($postsContainer[0]);
          }
          
          // Atualizar target de load more se houver next_cursor
          if (response.data.next_cursor) {
            const $loadMoreTarget = $('#load-more-posts-target');
            if ($loadMoreTarget.length) {
              $loadMoreTarget.find('#load-more-posts').attr('data-cursor', response.data.next_cursor);
            } else {
              // Se não existe, criar o target após os posts
              const loadMoreHtml = '<div id="load-more-posts-target" class="w-100 view-photo text-center mt-2 pointer">' +
                '<p id="load-more-posts" data-username="' + username + '" data-product="' + productId + '" data-cursor="' + response.data.next_cursor + '">Mostrar mais postagens</p>' +
                '</div>';
              $postsContainer.after(loadMoreHtml);
            }
          } else {
            // Remover botão "Ver mais" se não houver mais conteúdo
            $('#load-more-posts-target').remove();
          }
          
          // Mostrar posts com animação
          $postsContainer.fadeIn(300);
          $loadingState.addClass('d-none');
          
          
          // Mostrar mensagem de sucesso
          const successMessage = response.data.message || 'Cache limpo e conteúdo atualizado!';
          if (typeof displaySuccessToast === 'function') {
            displaySuccessToast(successMessage);
          } else {
            displayErrorToast(successMessage);
          }
        } else {
          const errorMsg = (response.data && response.data.message) ? response.data.message : 'Erro ao limpar cache';
          displayErrorToast(errorMsg);
          $postsContainer.fadeIn(300);
          $loadingState.addClass('d-none');
        }
      },
      error: function (xhr, status, error) {
        console.error('Erro ao limpar cache:', xhr, status, error);
        let errorMessage = 'Erro ao limpar cache';
        if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
          errorMessage = xhr.responseJSON.data.message;
        } else if (error) {
          errorMessage = 'Erro ao limpar cache: ' + error;
        }
        displayErrorToast(errorMessage);
        $postsContainer.fadeIn(300);
        $loadingState.addClass('d-none');
      },
      complete: function () {
        // Reabilitar botão
        $btn.prop('disabled', false);
        $btnText.show();
        $btnLoading.hide();
      }
    });
  });

  $(document).on("click", "#load-more-posts", function () {
    if (loading) return;

    const data = Object.fromEntries(flowCache.get("startFormData"));

    const el = $(this);
    const originalContent = el.html();

    loading = true;
    el.html(
      '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span>'
    );

    $.ajax({
      url: ajax_object.ajax_url,
      type: "POST",
      dataType: "json",
      data: {
        action: "load_more_posts",
        security: ajax_object.security,
        product_id: el.data("product"),
        end_cursor: el.data("cursor"),
        username: el.data("username") || data.username.toString(),
      },
      success: function (response) {
        loading = false;
        if (response.success) {
          $("#load-more-posts-target").replaceWith(response.data.html);
        } else {
          displayErrorToast(response.data.message);
        }
      },
      error: function (xhr, status, error) {
        loading = false;
        el.html(originalContent);
        alert("Erro ao processar a solicitação: " + error);
      },
    });
  });

  $(document).on("click", "#finish-action", function (e) {
    e.preventDefault();

    // Verificar se estamos na modal de pagamento
    // O botão está na modal de pagamento se estiver dentro de #paymentModal E não tiver data-service-type
    const isInPaymentModal = $(this).closest("#paymentModal").length > 0;
    const hasServiceType = $(this).data("service-type") !== undefined;
    
    // Se estiver na modal de pagamento (dentro de #paymentModal e sem data-service-type), processar pagamento
    if (isInPaymentModal && !hasServiceType) {
      const startLoading = () => {
        $(this).attr("disabled", "true");
        $(this).append(
          '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span>'
        );
      };
      const endLoading = () => {
        $(this).removeAttr("disabled");
        $(this).find(".spinner-border").remove();
      };
      
      startLoading();
      processPayment(endLoading);
      return;
    }

    // Se não estiver na modal de pagamento, verificar se precisa salvar posts
    const isCurtidas = $(this).data("service-type") == "curtidas";
    const isVisualizacoes = $(this).data("service-type") == "visualizacoes";
    const isVisualizacoesStory = $(this).data("service-type") == "visualizacoes_story";
    const shouldSaveSelectedPosts = isCurtidas || isVisualizacoes || isVisualizacoesStory;

    const startLoading = () => {
      $(this).attr("disabled", "true");
      $(this).append(
        '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span>'
      );
    };
    const endLoading = () => {
      $(this).removeAttr("disabled");
      $(this).find(".spinner-border").remove();
    };

    if (shouldSaveSelectedPosts) {
      const postsIds = Array.from($("[name='post']"), (it) => it.value);
      if (!postsIds.length) {
        return alert("Selecione pelo menos 1 post!");
      }

      startLoading();
      $.ajax({
        url: ajax_object.ajax_url,
        type: "POST",
        dataType: "json",
        data: {
          action: "save_selected_posts",
          security: ajax_object.security,
          product_id: $(this).data("product-id"),
          variation_id: $(this).data("variation-id"),
          posts_ids: postsIds,
        },
        success: function (json) {
          if (json.success) {
            $("#modal-container").find("#selectPostsModal")?.remove();
            $("#modal-container").append(json.data.html);
            
            // Verificar se o botão Limpar Cache foi adicionado
            // Botão Limpar Cache será detectado automaticamente via MutationObserver

            // HANDLE CPF VISIBILITY - Aguardar um pouco para garantir que o DOM foi atualizado
            setTimeout(function() {
              if (
                $('input[name="payment_method"][value="paghiper_pix"]').is(
                  ":checked"
                )
              ) {
                $("#upgram-cpf").removeClass("d-none");
              }
              $('input[name="payment_method"]').change(function () {
                if ($(this).val() === "paghiper_pix" && $(this).is(":checked")) {
                  $("#upgram-cpf").removeClass("d-none");
                } else {
                  $("#upgram-cpf").addClass("d-none");
                }
              });
            }, 100);
            
            endLoading();
          } else {
            endLoading();
            displayErrorToast(json.data.message);
          }
          // NÃO processar pagamento aqui - aguardar usuário clicar em "Realizar pagamento" na modal de pagamento
        },
        error: function (xhr, status, error) {
          endLoading();
          alert("Erro ao processar a solicitação: " + error);
        },
      });
    }
    // senão, vai direto pro qrcode
    else {
      startLoading();
      processPayment(endLoading);
    }
  });

  $(".upgram-content-trigger").on("click", function () {
    const contentId = $(this).data("tab");
    $(".upgram-product-section").attr("data-state", "inactive");
    $(`#${contentId}`).attr("data-state", "active");

    $(".upgram-product-description-button").attr("data-state", "inactive");
    $(this).attr("data-state", "active");
    
    // Lazy load do vídeo quando a aba de vídeo for clicada
    if (contentId === "upgram-section-video") {
      const videoContainer = $("#upgram-section-video .upgram-video-container");
      if (videoContainer.length && !videoContainer.data("loaded")) {
        const videoUrl = videoContainer.data("video-url");
        const placeholder = videoContainer.find(".upgram-video-placeholder");
        const loader = videoContainer.find(".upgram-video-loader");
        
        if (videoUrl && placeholder.length) {
          // Mostrar loader
          placeholder.hide();
          if (loader.length) {
            loader.show();
          }
          
          // Criar iframe após um pequeno delay para garantir que o loader apareça
          setTimeout(function() {
            const iframe = $('<iframe>', {
              src: videoUrl,
              frameborder: 0,
              allowfullscreen: true,
              style: 'position: absolute; top: 0; left: 0; width: 100%; height: 100%;',
              loading: 'lazy'
            });
            
            // Quando o iframe carregar, esconder o loader
            iframe.on('load', function() {
              if (loader.length) {
                loader.fadeOut(300);
              }
            });
            
            videoContainer.append(iframe);
            videoContainer.data("loaded", true);
          }, 100);
        }
      }
    }
    
    // Lazy load das avaliações quando a aba de avaliações for clicada
    if (contentId === "upgram-section-reviews") {
      const reviewsSection = $("#upgram-section-reviews");
      if (reviewsSection.length && !reviewsSection.data("loaded")) {
        const productId = reviewsSection.data("product-id");
        const loader = reviewsSection.find(".upgram-reviews-loader");
        const content = reviewsSection.find(".upgram-reviews-content");
        
        if (productId && loader.length && content.length) {
          // Mostrar loader
          loader.show();
          content.hide();
          
          // Carregar avaliações via AJAX
          $.ajax({
            url: ajax_object.ajax_url,
            type: 'POST',
            data: {
              action: 'upgram_load_reviews',
              product_id: productId,
              nonce: ajax_object.security
            },
            success: function(response) {
              if (response.success && response.data && response.data.html) {
                // Inserir HTML das avaliações
                content.html(response.data.html);
                content.show();
                loader.fadeOut(300);
                reviewsSection.data("loaded", true);
                
                // Re-inicializar scripts que possam ser necessários para as avaliações
                // (por exemplo, scripts do plugin de avaliações)
                if (typeof jQuery !== 'undefined' && jQuery.fn.ready) {
                  jQuery(document).trigger('upgram_reviews_loaded');
                }
              } else {
                loader.html('<div style="text-align: center; padding: 20px; color: #666;">Erro ao carregar avaliações. Por favor, recarregue a página.</div>');
              }
            },
            error: function() {
              loader.html('<div style="text-align: center; padding: 20px; color: #666;">Erro ao carregar avaliações. Por favor, recarregue a página.</div>');
            }
          });
        }
      }
    }
  });

  $(".upgram-link-trigger").on("click", function () {
    const href = $(this).data("href");
    window.location.href = href;
  });

  $("#seguidoresMenu")
    .find("a")
    .click(function (e) {
      e.preventDefault();
      const item = $(this).data("item");
      $("#seguidoresText").html(
        item.name +
          `<span class="purple"> ${Number(item.price).toLocaleString([], {
            style: "currency",
            currency: "BRL",
          })}</span>`
      );
      $("#productId").val(item.product_id);
      $("#variationId").val(item.variation_id);

      // update description
      const container = $("#upgram-variation-description");
      const templateHtml = $("#upgram-variation-description-template").html();
      container.empty();
      const lines = item.description?.split("\n") ?? [];
      $.each(lines, function (_, line) {
        line = $.trim(line);
        if (!line) return;
        const element = $(templateHtml).clone();
        element.find("span").text(line);
        container.append(element);
      });
    });

  function processPayment(onError) {
    // Coletar dados do formulário de pagamento
    const paymentMethod = $("#upgram-payment-method input:checked").val();
    const email = $('.upgram-input input[name="email"]').val() || $('#paymentModal input[name="email"]').val() || '';
    const phone = $('.upgram-input input[name="phone"]').val() || $('#paymentModal input[name="phone"]').val() || '';
    const cpf = $("#upgram-cpf input").val() || '';
    const createAccount = $("#upgram-create-account input:checked").val() || 'no';
    
    // Validar método de pagamento
    if (!paymentMethod) {
      displayErrorToast('Por favor, selecione um método de pagamento');
      if (onError) onError('Método de pagamento não selecionado');
      return;
    }
    
    // Validar e-mail antes de enviar (se a modal de pagamento estiver visível)
    if ($("#paymentModal").is(':visible') || $("#paymentModal").length > 0) {
      if (!email || email.trim() === '') {
        displayErrorToast('Por favor, preencha o e-mail antes de continuar');
        if (onError) onError('E-mail não preenchido');
        return;
      }
    }
    
    $.ajax({
      url: ajax_object.ajax_url,
      type: "POST",
      dataType: "json",
      data: {
        action: "process_payment",
        security: ajax_object.security,
        payment_method: paymentMethod,
        cpf: cpf,
        email: email,
        phone: phone,
        create_account: createAccount,
      },
      success: function (json) {
        if (json.success) {
          if (json.data.html) {
            $("#modal-container").html(json.data.html);
          } else {
            window.location.href = json.data.redirect_url;
          }
        } else {
          onError(json.data);
          displayErrorToast(json.data.message);
        }
      },
      error: function (xhr, status, error) {
        if (onError) onError(error);
        alert("Erro ao processar a solicitação: " + error);
      },
    });
  }

  $(document).on("click", "#instructions-back", function () {
    $("#modal-container").find('input[name="skip"]').val("false");
    $("#modal-container").find("#instructions").hide();
    $("#modal-container").find("#instagram-form").show();
  });

  $(document).on("click", "#back-to-start", function () {
    $("#modal-container").find("#paymentModal")?.remove();
    $("#modal-container").find("#selectPostsModal")?.remove();

    $("#modal-container").find("#initialModal").show();
  });

  $(document).on("click", "#upgram-coupon-button", function () {
    const showed = $("#couponForm").length;
    console.log("showed", showed);
    if (showed) return;

    const html = $("#template-coupon").html();
    $(this).after(html);

    $("#couponForm").on("submit", function (event) {
      event.preventDefault();
      const couponCode = $("#couponInput").val();

      if (couponCode.trim() === "") {
        alert("Por favor, insira um código de cupom.");
        return;
      }

      $.ajax({
        url: ajax_object.ajax_url,
        type: "POST",
        dataType: "json",
        data: {
          action: "validate_coupon",
          security: ajax_object.security,
          coupon_code: couponCode,
        },
        success: function (response) {
          if (response.success) {
            $("#upgram-coupon-placeholder").text(couponCode.toUpperCase());
            // Atualiza o valor total no frontend
            document.getElementById("totalValue").innerText =
              "R$" + response.data.discounted_total;
            // Exibe a mensagem de sucesso
            document.getElementById("couponMessage").innerHTML =
              "Cupom aplicado com sucesso, seu desconto foi de " +
              `<span>- R$ ${response.data.discount_amount}</span>`;
          } else {
            // Exibe a mensagem de erro
            document.getElementById("couponMessage").innerText =
              response.data.message;
          }
        },
        error: function (xhr, status, error) {
          alert("Erro ao processar a solicitação: " + error);
        },
      });
    });
  });

  function displayErrorToast(message) {
    $("#errorToast .toast-body").text(message);
    $("#upgram-toast-container").show();

    const toastElement = new bootstrap.Toast($("#errorToast")[0]);
    toastElement.show();

    $("#errorToast").on("hidden.bs.toast", function () {
      $("#upgram-toast-container").hide();
    });
  }
  function displaySuccessToast(message) {
    $("#successToast .toast-body").text(message);
    $("#upgram-toast-container").show();

    const toastElement = new bootstrap.Toast($("#successToast")[0]);
    toastElement.show();

    $("#successToast").on("hidden.bs.toast", function () {
      $("#upgram-toast-container").hide();
    });
  }

  window.resendOrderItem = async function resendOrderItem(
    orderId,
    itemId,
    onStart,
    onFinish
  ) {
    onStart();
    try {
      const { html } = await ajax({
        action: "resend_order_item",
        order_id: orderId,
        item_id: itemId,
      });
      $(`#order-${orderId}`).replaceWith(html);
      displaySuccessToast("Reenviado!");
    } finally {
      onFinish();
    }
  };


});

function verifyPayment(orderId, intervalId) {
  const $ = jQuery;
  $.ajax({
    url: ajax_object.ajax_url,
    type: "POST",
    dataType: "json",
    data: {
      action: "verify_payment",
      security: ajax_object.security,
      order_id: orderId,
    },
    success: function (response) {
      if (response.success) {
        const isPaid = response.data.is_paid;
        if (isPaid) {
          clearInterval(intervalId);
          loadSuccessModal(orderId);
        }
      } else {
        alert("Erro ao processar o pagamento: " + response.data.message);
      }
    },
    error: function (xhr, status, error) {
      alert("Erro ao processar a solicitação: " + error);
    },
  });
}

function loadSuccessModal(orderId) {
  const $ = jQuery;
  $.ajax({
    url: ajax_object.ajax_url,
    type: "POST",
    dataType: "json",
    data: {
      action: "load_success_modal",
      security: ajax_object.security,
      order_id: orderId,
    },
    success: function (response) {
      if (response.success) {
        $("#modal-container").html(response.data.html).modal("show");
      }
    },
  });
}

function copiarPix(qrCode) {
  navigator.clipboard.writeText(qrCode).then(() => alert("Pix copiado!"));
}

const masks = {
  cpf(input) {
    let value = input.value.replace(/\D/g, "");
    if (value.length > 11) value = value.slice(0, 11);
    value = value.replace(/(\d{3})(\d)/, "$1.$2");
    value = value.replace(/(\d{3})(\d)/, "$1.$2");
    value = value.replace(/(\d{3})(\d{1,2})$/, "$1-$2");
    input.value = value;
  },
};

/**
 * Tabela de Conteúdo (TOC) - Toggle simples
 */
(function() {
    'use strict';
    
    // Event delegation no document - funciona sempre
    document.addEventListener('click', function(e) {
        // Verificar se o clique foi em um botão de toggle do TOC
        var target = e.target;
        if (!target || !target.hasAttribute || !target.hasAttribute('data-upgram-toc-toggle')) {
            return;
        }
        
        e.preventDefault();
        e.stopPropagation();
        
        // Encontrar o TOC pai
        var toc = target.closest('[data-upgram-toc]');
        if (!toc) {
            return;
        }
        
        // Alternar classe
        toc.classList.toggle('upgram-toc-collapsed');
        
        // Atualizar texto do botão
        var collapsed = toc.classList.contains('upgram-toc-collapsed');
        target.textContent = collapsed ? 'Mostrar sumário' : 'Ocultar sumário';
    });
    
    // Função para atualizar textos iniciais
    function updateButtons() {
        var buttons = document.querySelectorAll('[data-upgram-toc-toggle]');
        for (var i = 0; i < buttons.length; i++) {
            var btn = buttons[i];
            var toc = btn.closest('[data-upgram-toc]');
            if (toc) {
                var collapsed = toc.classList.contains('upgram-toc-collapsed');
                btn.textContent = collapsed ? 'Mostrar sumário' : 'Ocultar sumário';
            }
        }
    }
    
    // Atualizar quando DOM estiver pronto
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', updateButtons);
    } else {
        updateButtons();
    }
    
    // Re-atualizar após delays para pegar conteúdo dinâmico
    setTimeout(updateButtons, 500);
    setTimeout(updateButtons, 1500);
})();
